__author__='Luk Peeters'
# BA_visualisation
"""
Module with basic BA visualisation functions
Based on original script by Simon Gallant
"""
import matplotlib.pyplot as plt
import matplotlib.font_manager as fnt
from matplotlib.ticker import FormatStrFormatter
from matplotlib.ticker import MaxNLocator

def BA_colors(picker=None, name=None, RGB=None):
    """
    This function is used for the definition of colo(u)rs used in Bioregional assessments.

    Two ways of identifying a color are available to you: *picker* and *name*.

    *picker* takes an integer, *name* takes a string.
    The table below shows which RGB value is returned for each.
    These RGBs are approved for use in Bioregional Assessments and are visually accessible.
    The order that they are listed here is the order in which they should be used.

    ======   ============   ============
    picker   name           RGB value
    ======   ============   ============
    1        Green          46,136,36
    2        Orange         207,111,10
    3        DarkBlue       79,110,161
    4        Slate          64,79,97
    5        Red            204,62,42
    6        LightBlue      35,158,215
    97       GreenShading   210,236,248
    98       BlueShading    210,236,248
    99       Grid           178,178,178
    ======   ============   ============

    Note that PyPlot takes RGB values between 0 and 1, so a conversion takes place before the output is returned.

    Additionaly, you can use this function to convert known RGB values to pyplot RGBs:

    * use the kwarg *RGB* to input a list in the format [R,G,B] where all of R, G and B are ints from 0 to 255.

    Examples:

    >>> BA_colors(picker=1)
    [0.1803921568627451, 0.5333333333333333, 0.1411764705882353]
    >>> BA_colors(name='LightBlue')
    [0.13725490196078433, 0.6196078431372549, 0.8431372549019608]
    >>> BA_colors(RGB=[159,89,43])
    [0.6235294117647059, 0.34901960784313724, 0.16862745098039217]
    """

    def RGBConv(RGB):
        R,G,B = RGB

        r = float(R)/255.0
        g = float(G)/255.0
        b = float(B)/255.0

        return [r,g,b]

    NameToCode = {'Green':1,
                  'Orange':2,
                  'DarkBlue':3,
                  'Slate':4,
                  'Red':5,
                  'LightBlue':6,
                  'GreenShading':97,
                  'BlueShading':98,
                  'Grid':99}

    CodeToRGB = {1:[46,136,36],
                 2:[207,111,10],
                 3:[79,110,161],
                 4:[64,79,97],
                 5:[204,62,42],
                 6:[35,158,215],
                 97:[205,241,202],
                 98:[210,236,248],
                 99:[178,178,178]}


    if name:
        rgb = RGBConv(CodeToRGB[NameToCode[name]])
    elif picker:
        rgb = RGBConv(CodeToRGB[picker])
    elif RGB:
        rgb = RGBConv(RGB)

    return rgb

def BA_font(s=8,fweight='normal'):
    """
    defines a specific set of font properties as used in Bioregional Assessment, specificially Arial 8.
    The size can be changed if required, for instance if using unicode characters.
    """

    fontProperties = fnt.FontProperties(family='Arial', weight=fweight, style = 'normal', size = s)
    return fontProperties

def adjust_spines(ax,spines):
    """
    Set the visibility of the spines in an axes.
    ax is the axes you wish to control, spines is string: 'left' or 'bottom'. 'leftbottom' also works.
    This will make visible only the spines you specify.
    Note that this function has been tweaked to work with this script, and may not function correctly in a generic environment.
    """
    for loc, spine in ax.spines.items():
        if loc in spines:
            spine.set_smart_bounds(True)
        else:
            spine.set_color('none') # don't draw spine
    ax.xaxis.tick_top()
    # turn off ticks where there is no spine
    if 'left' in spines:
        ax.yaxis.set_ticks_position('left')

    if 'bottom' in spines:
        ax.xaxis.set_ticks_position('bottom')

    else:
        # no xaxis ticks
        ax.xaxis.set_ticks([])

def customboxplot(dat, vertset=False):
    plt.boxplot(dat,
                whis=[5, 95],
                sym='.',
                vert=vertset,
                boxprops=dict(color='k'),
                flierprops=dict(ms=1, color=[0.25, 0.25, 0.25, 1]),
                whiskerprops=dict(color='k', linestyle='-'),
                capprops=dict(color='k'),
                medianprops=dict(color=BA_colors(name='Red')))

letters = 'abcdefghijklmnopqurtuvwxyz'
